/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef MX__REQUEST_H
# define MX__REQUEST_H

#include "mx__valgrind.h"
#include "mx__wait_queue.h"
#include "mx__lib.h"

void mx__rl__alloc(struct mx__request_lookaside *rl);
int mx__rl_init(struct mx__request_lookaside *rl);
void mx__rl_fini(struct mx__request_lookaside *rl);

uint16_t mx_checksum(void *ptr, unsigned length);


/*********
 * Routines for basic request manipulation
 */

#define MX__REQUEST_FROM_QUEUE_ELT(x)			\
 ((union mx_request *)					\
   (((char*) x)						\
    - offsetof(union mx_request, basic.queue_elt)))

static inline void
mx__init_request_queue(struct mx__request_queue_head * head)
{
  head->prev = head->next = head;
}

#define mx__isempty_request_queue(head) (head == (head)->next)

static inline void
mx__enqueue_request(struct mx__request_queue_head * head, union mx_request *req)
{
  struct mx__request_queue_head * elt = &req->basic.queue_elt;
#if MX_DEBUG
  mx_assert(elt->prev == NULL);
  mx_assert(elt->next == NULL);
#endif
  elt->next = head;
  elt->prev = head->prev;
  head->prev->next = elt;
  head->prev = elt;
}

static inline void
mx__putback_request(struct mx__request_queue_head * head, union mx_request *req)
{
  struct mx__request_queue_head * elt = &req->basic.queue_elt;
#if MX_DEBUG
  mx_assert(elt->prev == NULL);
  mx_assert(elt->next == NULL);
#endif
  elt->next = head->next;
  elt->prev = head;
  head->next->prev = elt;
  head->next = elt;
}

static inline union mx_request *
mx__first_request(struct mx__request_queue_head * head)
{
  mx_assert(head != head->next);
  return MX__REQUEST_FROM_QUEUE_ELT(head->next);
}

static inline union mx_request *
mx__dequeue_request(struct mx__request_queue_head * head)
{
  struct mx__request_queue_head * elt;
  union mx_request *req;

  mx_assert(head != head->next);
  elt = head->next;
  req = MX__REQUEST_FROM_QUEUE_ELT(elt);
  head->next = elt->next;
  head->next->prev = head;
#if MX_DEBUG
  elt->prev = elt->next = NULL;
#endif
  return req;
}

#define MX__FOREACH_REQ(req, elt, head)				\
for(elt = (head)->next, req = MX__REQUEST_FROM_QUEUE_ELT(elt);	\
    elt != head;						\
    elt = elt->next, req = MX__REQUEST_FROM_QUEUE_ELT(elt))

#define MX__FOREACH_REQ_SAFE(req, elt, next, head)					\
for(elt = (head)->next, next = elt->next, req = MX__REQUEST_FROM_QUEUE_ELT(elt);	\
    elt != head;									\
    elt = next, next = elt->next, req = MX__REQUEST_FROM_QUEUE_ELT(elt))

static inline void
mx__spliceout_request(struct mx__request_queue_head * head, union mx_request *req)
{
  struct mx__request_queue_head * elt;
#if MX_DEBUG
  union mx_request * r;
  MX__FOREACH_REQ(r, elt, head)
    if (r == req)
      break;
  mx_always_assert(r == req);
#endif
  elt = &req->basic.queue_elt;
  elt->next->prev = elt->prev;
  elt->prev->next = elt->next;
#if MX_DEBUG
  elt->prev = elt->next = NULL;
#endif
}

static inline int
mx__count_requests(struct mx__request_queue_head * head)
{
  struct mx__request_queue_head * elt;
  int count;
  for(elt = head->next, count = 0; elt != head; elt = elt->next, count++);
  return count;
}

static inline union mx_request *
mx__find_receive_by_match_info(struct mx__request_queue_head * head,
				    uint64_t match_info)
{
  union mx_request * r;
  struct mx__request_queue_head * elt;
  MX__FOREACH_REQ(r, elt, head)
    if (r->recv.match_info == (match_info & r->recv.match_mask))
      return r;
  return NULL;
}

static inline union mx_request *
mx__find_request_by_matching(struct mx__request_queue_head * head,
			     uint64_t match_info,
			     uint64_t match_mask)
{
  union mx_request *r;
  struct mx__request_queue_head * elt;
  MX__FOREACH_REQ(r, elt, head)
    /* The matching looks backwards compared to match_recv_request. */
    if ((r->basic.status.match_info & match_mask) == match_info)
      return r;
  return NULL;
}

/*********
 * Routines to manipulate partner-queued requests
 */

#define MX__REQUEST_FROM_PARTNER_QUEUE_ELT(x)			\
 ((union mx_request *)						\
  (((char*) x)							\
   - offsetof(union mx_request, basic.partner_queue_elt)))

static inline void
mx__init_partner_request_queue(struct mx__partner_request_queue_head * head)
{
  head->prev = head->next = head;
}

#define mx__isempty_partner_request_queue(head) (head == (head)->next)

static inline void
mx__enqueue_partner_request(struct mx__partner_request_queue_head * head,
			    union mx_request *req)
{
  struct mx__partner_request_queue_head * elt = &req->basic.partner_queue_elt;
#if MX_DEBUG
  mx_assert(elt->prev == NULL);
  mx_assert(elt->next == NULL);
#endif
  elt->next = head;
  elt->prev = head->prev;
  head->prev->next = elt;
  head->prev = elt;
}

static inline union mx_request *
mx__first_partner_request(struct mx__partner_request_queue_head * head)
{
  mx_assert(head != head->next);
  return MX__REQUEST_FROM_PARTNER_QUEUE_ELT(head->next);
}

static inline union mx_request *
mx__dequeue_partner_request(struct mx__partner_request_queue_head * head)
{
  struct mx__partner_request_queue_head * elt;
  union mx_request *req;

  mx_assert(head != head->next);
  elt = head->next;
  req = MX__REQUEST_FROM_PARTNER_QUEUE_ELT(elt);
  head->next = elt->next;
  head->next->prev = head;
#if MX_DEBUG
  elt->prev = elt->next = NULL;
#endif
  return req;
}

#define MX__FOREACH_PARTNER_REQ(req, elt, head)				\
for(elt = (head)->next, req = MX__REQUEST_FROM_PARTNER_QUEUE_ELT(elt);	\
    elt != head;							\
    elt = elt->next, req = MX__REQUEST_FROM_PARTNER_QUEUE_ELT(elt))

#define MX__FOREACH_PARTNER_REQ_SAFE(req, elt, next, head)					\
for(elt = (head)->next, next = elt->next, req = MX__REQUEST_FROM_PARTNER_QUEUE_ELT(elt);	\
    elt != head;										\
    elt = next, next = elt->next, req = MX__REQUEST_FROM_PARTNER_QUEUE_ELT(elt))

static inline void
mx__spliceout_partner_request(struct mx__partner_request_queue_head * head,
			      union mx_request *req)
{
  struct mx__partner_request_queue_head * elt;
#if MX_DEBUG
  union mx_request * r;
  MX__FOREACH_PARTNER_REQ(r, elt, head)
    if (r == req)
      break;
  mx_always_assert(r == req);
#endif
  elt = &req->basic.partner_queue_elt;
  elt->next->prev = elt->prev;
  elt->prev->next = elt->next;
#if MX_DEBUG
  elt->prev = elt->next = NULL;
#endif
}

static inline
union mx_request *
mx__find_partner_receive_by_msg_seq(struct mx__partner_request_queue_head * head,
				    uint64_t msg_seq)
{
  struct mx__partner_request_queue_head * elt;
  union mx_request * r;
  MX__FOREACH_PARTNER_REQ(r, elt, head)
    if (msg_seq == r->recv.msg_seq)
      return r;
  return NULL;
}

static inline void
mx__replace_partner_request(struct mx__partner_request_queue_head * head,
			    union mx_request * old, union mx_request * new)
{
  struct mx__partner_request_queue_head * old_elt = &old->basic.partner_queue_elt;
  struct mx__partner_request_queue_head * new_elt = &new->basic.partner_queue_elt;
  new_elt->prev = old_elt->prev;
  new_elt->next = old_elt->next;
  new_elt->next->prev = new_elt;
  new_elt->prev->next = new_elt;
}

/*******
 * Routines used on request completion
 */

static inline void
mx__notify_waiter_request_done(struct mx_endpoint *ep, union mx_request *req)
{
  if (req->basic.wq && req->basic.wq->wait_state == MX__WAIT_STATE_WAITING) {
    req->basic.wq->wait_state = MX__WAIT_STATE_COMPLETE;
    mx__wait_queue_spliceout(&ep->wait_queue_head, req->basic.wq);
    ep->wait_waiters--;
    MX__EVENT_SIGNAL(&req->basic.wq->event);
  }
}

static inline void
mx__notify_peeker_request_done(struct mx_endpoint *ep, union mx_request *req, uint32_t ctxid)
{
  mx__enqueue_request(&ep->ctxid[ctxid].doneq, req);

  if (ep->peek_waiters && !mx__wait_queue_isempty(&ep->ctxid[ctxid].peek_queue_head)) {
    struct mx__wait_queue_head * elt;
    struct mx__wait_queue * wq;
    /* wake up the first matching */
    MX__FOREACH_WAIT_QUEUE(wq, elt, &ep->ctxid[ctxid].peek_queue_head)
      if ((req->basic.status.match_info & wq->type.peek.match_mask)
	  == wq->type.peek.match_info) {
	mx__wait_queue_spliceout (&ep->ctxid[ctxid].peek_queue_head, wq);
	wq->wait_state = MX__WAIT_STATE_COMPLETE;
	wq->result = 1;
	wq->type.peek.request = req;
	if (wq->type.peek.is_wait_any) {
	  mx__spliceout_request(&ep->ctxid[ctxid].doneq, req);
	  req->basic.acquired_by_wait_any = 1;
	  ep->acquired_by_wait_any_count++;
	}
	ep->peek_waiters--;
	MX__EVENT_SIGNAL(&wq->event);
	break; /* only wakeup one peeker */
      }
  }
}

static inline void
mx__notify_prober_unexpected(struct mx_endpoint *ep, union mx_request *req, uint32_t ctxid)
{
  mx__enqueue_request(&ep->ctxid[ctxid].unexpq, req);

  if (ep->probe_waiters && !mx__wait_queue_isempty(&ep->ctxid[ctxid].probe_queue_head)) {
    struct mx__wait_queue * wq = mx__try_probe(ep, ctxid);
    if (wq != NULL) {
      wq->wait_state = MX__WAIT_STATE_COMPLETE;
      wq->result = 1;
      wq->type.probe.request = req;
      mx__wait_queue_spliceout (&ep->ctxid[ctxid].probe_queue_head, wq);
      ep->probe_waiters--;
      MX__EVENT_SIGNAL(&wq->event);
    }
  }
}


/***********
 * Request allocation
 */

#ifndef MX_KERNEL
#define REQ_CHUNK 1024
#else
#define REQ_CHUNK (4000/sizeof(union mx_request) - 1)
#endif

static inline union mx_request *
mx__rl_alloc(struct mx_endpoint *ep)
{
  struct mx__request_lookaside *rl = &ep->req_lookaside;
  union mx_request * r;

  if (unlikely(rl->count == 0)) {
    mx__rl__alloc(rl);
    if (unlikely(rl->count == 0))
      return NULL;
  }

  r = MX__REQUEST_FROM_QUEUE_ELT(rl->free_requests_queue.next);
  rl->free_requests_queue.next = rl->free_requests_queue.next->next;
  rl->count--;

  /* debugging initializations */
  MX_VALGRIND_MEMORY_MAKE_WRITABLE(r, sizeof(*r));
#if MX_DEBUG
#if !MX_VALGRIND_DEBUG
  memset(r, 0, sizeof(*r));
#endif
  r->basic.queue_elt.next = r->basic.queue_elt.prev = NULL;
  r->basic.partner_queue_elt.next = r->basic.partner_queue_elt.prev = NULL;
  r->basic.state = 0;
#endif

  /* mandatory initializations */
  r->basic.status.code = MX_STATUS_SUCCESS;
  r->basic.timeout = ep->timeout;

  return r;
}

static inline void 
mx__rl_free(struct mx_endpoint *ep, union mx_request *r)
{
  struct mx__request_lookaside *rl = &ep->req_lookaside;
  struct mx__request_queue_head * elt = &r->basic.queue_elt;

  mx_assert(elt->next == NULL);
  mx_assert(elt->prev == NULL);
  MX_VALGRIND_MEMORY_MAKE_NOACCESS(r, sizeof(*r));
  MX_VALGRIND_MEMORY_MAKE_WRITABLE(&elt->next,
				   sizeof(elt->next));
  elt->next = rl->free_requests_queue.next;
  rl->free_requests_queue.next = elt;
  if (MX_DEBUG) {
    MX_VALGRIND_MEMORY_MAKE_WRITABLE(&r->basic.state, sizeof(r->basic.state));
    r->basic.state = MX__REQUEST_STATE_FREED;
  }
  rl->count++;
}

#endif /* REQUEST_H */
